﻿using System.Collections.Generic;
using static Globales.FileTypes.TipoFichero;

namespace Globales{
	public static class FileTypes{
		public enum ClaseFichero : byte{
			AerotriVarios,
			Fotogramas,
			Camara,
			Puntos,
			Orientaciones,
			OrientacionesyPuntos,
			GPS,
			Otros,
		}
		public enum TipoFichero : int{
			AerotriTrabajo,
			AerotriFotos,
			PATBFotos,
			ImasterFotos,
			EMosaicFotos,
			AerotriCamara,
			IniCamara,
			ImasterCamara,
			RapidCal,
			DigiCamara,
			AgisoftCamara,
			AerotriApoyo,
			XYZ,
			Aerotriccpp,	//Ficheros con coordenadas (ajustadas o no) de cc.pp. También se pueden
			ΩΦΚ,			//emplear como de GPS, así que se colocan justo antes que ellos
			PATBori,
			IMasterext,
			AerotriGps,		//Aquí empiezan los de navegación
			MatchATGps,
			TopoSysGps,
			IMasterpairlist,
			IMasterrel,

			AllFiles,	//pseudo file-type
		}

		public enum TipoFiltro : byte{
			TrabajoAerotri,
			FotogramasLeer,Aproximadas,Apoyo,PuntosTodos,
			GPSLeer,InternaLeer,
			FotogramasEscribir,GPSTodosLeer,GPSTodosEscribir,
			InternaEscribir
		}

		private const byte RW=3, R=1, W=3;

		public struct FileTypeInfo{
			public TipoFichero filetype;
			public ClaseFichero clase;
			public byte IO;		//rw. El bit bajo indica si Aerotri es capaz de leer este tipo de fichero; el segundo bit, si es capaz de escribirlo
			public string[] extensiones;
			public byte aerotri;	//Código que identifica este formato en las librerías de Aerotri
			public FileTypeInfo(TipoFichero t, ClaseFichero c, byte io, string[] e, byte aerotri_code){
				filetype=t;
				clase=c;
				IO=io;
				extensiones=e;
				aerotri=aerotri_code;
			}
		}

		public static readonly Dictionary<TipoFichero,FileTypeInfo> FileTypeInfos;
		public static readonly Dictionary<TipoFiltro,TipoFichero[]> Filtros;

		static void AddFileType(TipoFichero t, ClaseFichero c, byte io, string[] e, byte aerotri_code){
			FileTypeInfos.Add(t,new FileTypeInfo(t,c,io,e,aerotri_code));
		}
		static FileTypes(){
			FileTypeInfos=new Dictionary<TipoFichero, FileTypeInfo>(40);
			AddFileType(AerotriTrabajo, ClaseFichero.AerotriVarios,RW,new string[]{".art"},0);
			AddFileType(AerotriFotos, ClaseFichero.Fotogramas,RW,new string[]{".ftm"},0);
			AddFileType(PATBFotos, ClaseFichero.Fotogramas,R,new string[]{".f",".ff"},1);
			AddFileType(ImasterFotos, ClaseFichero.Fotogramas,R,new string[]{".imc"},2);
			AddFileType(EMosaicFotos, ClaseFichero.Fotogramas,R,new string[]{".pts"},3);
			AddFileType(AerotriCamara, ClaseFichero.Camara,RW,new string[]{".int"},0);
			AddFileType(IniCamara, ClaseFichero.Camara,RW,new string[]{".ini"},1);
			AddFileType(ImasterCamara, ClaseFichero.Camara,RW,new string[]{".cmr"},2);
			AddFileType(RapidCal, ClaseFichero.Camara,R,new string[]{".cal"},3);
			AddFileType(DigiCamara, ClaseFichero.Camara,RW,new string[]{".cam"},4);
			AddFileType(AgisoftCamara, ClaseFichero.Camara,W,new string[]{".cam"},5);
			AddFileType(AerotriApoyo, ClaseFichero.Puntos,RW,new string[]{".pym"},1);
			AddFileType(XYZ, ClaseFichero.Puntos,RW,new string[]{".XYZ",".txt"},0);
			AddFileType(Aerotriccpp, ClaseFichero.OrientacionesyPuntos,RW,new string[]{".prm",".ajs"},1);
			AddFileType(ΩΦΚ, ClaseFichero.Orientaciones,RW,new string[]{".eo"},2);
			AddFileType(PATBori, ClaseFichero.Orientaciones,RW,new string[]{".ori"},3);
			AddFileType(IMasterext, ClaseFichero.Orientaciones,W, new string[]{".ext"},4);
			AddFileType(AerotriGps, ClaseFichero.GPS,RW, new string[]{".gpn"},5);
			AddFileType(MatchATGps, ClaseFichero.GPS,R, new string[]{".exp"},6);
			AddFileType(TopoSysGps, ClaseFichero.GPS,R, new string[]{".txt"},7);
			AddFileType(IMasterpairlist, ClaseFichero.Otros,W, new string[]{".pairlist"},4);	//El mismo que los .ext
			AddFileType(IMasterrel, ClaseFichero.Otros,W,new string[]{".rel"},100);

			Filtros=new Dictionary<TipoFiltro, TipoFichero[]>(20);
			Filtros.Add(TipoFiltro.TrabajoAerotri,new TipoFichero[]{AerotriTrabajo});
			Filtros.Add(TipoFiltro.FotogramasLeer,new TipoFichero[]{AerotriFotos,PATBFotos,ImasterFotos,EMosaicFotos});
			Filtros.Add(TipoFiltro.Aproximadas,new TipoFichero[]{Aerotriccpp});
			Filtros.Add(TipoFiltro.Apoyo,new TipoFichero[]{AerotriApoyo,XYZ});
			Filtros.Add(TipoFiltro.PuntosTodos,new TipoFichero[]{Aerotriccpp,AerotriApoyo,XYZ});
			Filtros.Add(TipoFiltro.GPSLeer,new TipoFichero[]{AerotriGps,ΩΦΚ,MatchATGps,TopoSysGps});
			Filtros.Add(TipoFiltro.InternaLeer,new TipoFichero[]{AerotriCamara,ImasterCamara,IniCamara,DigiCamara,RapidCal});
			Filtros.Add(TipoFiltro.FotogramasEscribir,new TipoFichero[]{AerotriFotos,PATBFotos,ImasterFotos});
			Filtros.Add(TipoFiltro.GPSTodosLeer,new TipoFichero[]{Aerotriccpp,ΩΦΚ,PATBori,AerotriGps,MatchATGps,TopoSysGps});
			Filtros.Add(TipoFiltro.GPSTodosEscribir,new TipoFichero[]{Aerotriccpp,ΩΦΚ,PATBori,AerotriGps,IMasterpairlist});
			Filtros.Add(TipoFiltro.InternaEscribir,new TipoFichero[]{AerotriCamara,ImasterCamara,IniCamara,DigiCamara,AgisoftCamara});

			Español=new Dictionary<TipoFichero, NombrecortoyLargo>(40);
			Inglés=new Dictionary<TipoFichero, NombrecortoyLargo>(40);
			Italiano=new Dictionary<TipoFichero, NombrecortoyLargo>(40);

			Nombres=new Dictionary<TipoFichero, NombrecortoyLargo>[(byte)Idioma.NIDIOMAS];
			Nombres[(byte)Idioma.Esp]=Español;
			Nombres[(byte)Idioma.Eng]=Inglés;
			Nombres[(byte)Idioma.It]=Italiano;
			

			foreach(TipoyNombres t in NombresIdiomas.Español) Español.Add(t.tipo,t.nombres);
			foreach(TipoyNombres t in NombresIdiomas.Inglés) Inglés.Add(t.tipo,t.nombres);
			foreach(TipoyNombres t in NombresIdiomas.Italiano) Italiano.Add(t.tipo,t.nombres);
		}

		internal struct NombrecortoyLargo{
			public string corto;
			public string largo;
		};
		internal struct TipoyNombres{
			public TipoFichero tipo;
			public NombrecortoyLargo nombres;
			public TipoyNombres(TipoFichero t, string c, string l){
				tipo=t;
				nombres.corto=c;
				nombres.largo=l;
			}
		}
		internal static readonly Dictionary<TipoFichero,NombrecortoyLargo> Español,Inglés,Italiano;
		internal static readonly Dictionary<TipoFichero,NombrecortoyLargo>[] Nombres;

		internal class NombresIdiomas{
			internal static readonly TipoyNombres[] Español={
				new TipoyNombres(AerotriTrabajo,"Trabajo de Aerotri","Trabajos de Aerotri"),
				new TipoyNombres(AerotriFotos,"Aerotri","Ficheros de fotocoordenadas de Aerotri"),
				new TipoyNombres(PATBFotos,"PATB","Ficheros de fotocoordenads de PATB"),
				new TipoyNombres(ImasterFotos,"Image Master","Ficheros de fotocoordenadas de Image Master"),
				new TipoyNombres(EMosaicFotos,"Enso Mosaic","Ficheros de fotocoordenadas de Enso Mosaic"),
				new TipoyNombres(AerotriCamara,"Aerotri","Ficheros de cámara de Aerotri"),
				new TipoyNombres(IniCamara,"ini","Ficheros de cámara de formato ini"),
				new TipoyNombres(ImasterCamara,"Image Master","Ficheros de cámara de Image Master"),
				new TipoyNombres(RapidCal,"RapidCal","Ficheros de cámara de RapidCal"),
				new TipoyNombres(DigiCamara,"Digi","Ficheros de cámara de Digi (nuevos)"),
				new TipoyNombres(AgisoftCamara,"Agisoft","Ficheros de cámara de Agisfot"),
				new TipoyNombres(AerotriApoyo,"Aerotri","Ficheros de apoyo de Aerotri"),
				new TipoyNombres(XYZ,"XYZ","Ficheros de puntos con líneas Nombre,X,Y,Z"),
				new TipoyNombres(Aerotriccpp,"Aerotri, cc.pp.","Ficheros de centros de proyección y puntos de Aerotri"),
				new TipoyNombres(ΩΦΚ,"XYZΩΦΚ","Ficheros de centros de proyección con líneas Nombre,X,Y,Z,Ω,Φ,Κ"),
				new TipoyNombres(PATBori,"PATB","Ficheros de orientaciones de PATB"),
				new TipoyNombres(IMasterext,"Image Master","Ficheros de orientación de Image Master"),
				new TipoyNombres(AerotriGps,"Aerotri, gps/ins","Ficheros de eventos de GPS/INS de Aerotri"),
				new TipoyNombres(MatchATGps,"MatchAT","Ficheros de eventos de GPS/INS de MatchAT"),
				new TipoyNombres(TopoSysGps,"EO de TopoSys","Ficheros de orientaciones de TopoSys"),
				new TipoyNombres(IMasterpairlist,"Image Master","Lista de pares de Image Master"),
				new TipoyNombres(IMasterrel,"Image Master","Orientaciones relativas de Image Master"),
				new TipoyNombres(AllFiles,"Otro","Todos los ficheros"),
			};

			internal static readonly TipoyNombres[] Inglés={
				new TipoyNombres(AerotriTrabajo,"Aerotri work","Aerotri works"),
				new TipoyNombres(AerotriFotos,"Aerotri","Aerotri photo coordinates files"),
				new TipoyNombres(PATBFotos,"PATB","PATB photo coordinates files"),
				new TipoyNombres(ImasterFotos,"Image Master","Image Master photo coordinates files"),
				new TipoyNombres(EMosaicFotos,"Enso Mosaic","Enso Mosaic photo coordinates files"),
				new TipoyNombres(AerotriCamara,"Aerotri","Aerotri camera files"),
				new TipoyNombres(IniCamara,"ini","Camera files with ini fomat"),
				new TipoyNombres(ImasterCamara,"Image Master","Image Master camera files"),
				new TipoyNombres(RapidCal,"RapidCal","RapidCal camera files"),
				new TipoyNombres(DigiCamara,"Digi","Digi camera files (new format)"),
				new TipoyNombres(AgisoftCamara,"Agisoft","Agisfot camera files"),
				new TipoyNombres(AerotriApoyo,"Aerotri","Aerotri control point files"),
				new TipoyNombres(XYZ,"XYZ","Point files with lines like Name,X,Y,Z"),
				new TipoyNombres(Aerotriccpp,"Aerotri, cc.pp.","Aerotri projection centers and points files"),
				new TipoyNombres(ΩΦΚ,"XYZΩΦΚ","Projection centers files with lines like: Name,X,Y,Z,Ω,Φ,Κ"),
				new TipoyNombres(PATBori,"PATB","PATB orientations files"),
				new TipoyNombres(IMasterext,"Image Master","Image Master orientation files"),
				new TipoyNombres(AerotriGps,"Aerotri, gps/ins","Aerotri GPS/INS events files"),
				new TipoyNombres(MatchATGps,"MatchAT","MATCH-AT GPS and INS events files"),
				new TipoyNombres(TopoSysGps,"TopoSys's EO","TopoSys orientations files"),
				new TipoyNombres(IMasterpairlist,"Image Master","Image Master pair lists"),
				new TipoyNombres(IMasterrel,"Image Master","Image Master relative orientations"),
				new TipoyNombres(AllFiles,"Other","All files"),
			};

			internal static readonly TipoyNombres[] Italiano={
				new TipoyNombres(AerotriTrabajo,"Lavoro di Aerotri","Lavori di Aerotri"),
				new TipoyNombres(AerotriFotos,"Aerotri","File di fotocoordinade di Aerotri"),
				new TipoyNombres(PATBFotos,"PATB","File di fotocoordinade di PATB"),
				new TipoyNombres(ImasterFotos,"Image Master","File di fotocoordinade di Image Master"),
				new TipoyNombres(EMosaicFotos,"Enso Mosaic","File di fotocoordinade di Enso Mosaic"),
				new TipoyNombres(AerotriCamara,"Aerotri","File di camera di Aerotri"),
				new TipoyNombres(IniCamara,"ini","File di camera di formato ini"),
				new TipoyNombres(ImasterCamara,"Image Master","File di camera di Image Master"),
				new TipoyNombres(RapidCal,"RapidCal","File di camera di RapidCal"),
				new TipoyNombres(DigiCamara,"Digi","File di camera di Digi (nuovi)"),
				new TipoyNombres(AgisoftCamara,"Agisoft","File di camera di Agisfot"),
				new TipoyNombres(AerotriApoyo,"Aerotri","File di appoggio di Aerotri"),
				new TipoyNombres(XYZ,"XYZ","File di punto con righe Nome,X,Y,Z"),
				new TipoyNombres(Aerotriccpp,"Aerotri, cc.pp.","File di centri di proiezione e punti di Aerotri"),
				new TipoyNombres(ΩΦΚ,"XYZΩΦΚ","File di centri di proiezione con righe Nome,X,Y,Z,Ω,Φ,Κ"),
				new TipoyNombres(PATBori,"PATB","File di orientazioni di PATB"),
				new TipoyNombres(IMasterext,"Image Master","File di orientazioni di Image Master"),
				new TipoyNombres(AerotriGps,"Aerotri, gps/ins","File di GPS/INS di Aerotri"),
				new TipoyNombres(MatchATGps,"MatchAT","File di GPS/INS di MatchAT"),
				new TipoyNombres(TopoSysGps,"EO de TopoSys","File di orientazioni di TopoSys"),
				new TipoyNombres(IMasterpairlist,"Image Master","Lista di coppie di Image Master"),
				new TipoyNombres(IMasterrel,"Image Master","Orientazioni relative di Image Master"),
				new TipoyNombres(AllFiles,"Altro","Tutti i file"),
			};
		}

		public static string GetName(TipoFichero tipo, Idioma idioma){
			if(idioma>Idioma.NIDIOMAS) idioma=Idioma.Esp;
			return Nombres[(byte)idioma][tipo].corto;
		}

		public static string[] GetShortNames(TipoFiltro tfiltro, Idioma idioma){
			if(idioma>Idioma.NIDIOMAS) idioma=Idioma.Esp;
			TipoFichero[] filtroftypes;
			Dictionary<TipoFichero,NombrecortoyLargo> nombres;
			string[] ss;

			nombres=Nombres[(byte)idioma];
			filtroftypes=Filtros[tfiltro];
			ss=new string[filtroftypes.Length];
			for(int i=0;i<filtroftypes.Length; i++){
				ss[i]=nombres[filtroftypes[i]].corto;
			}
			return ss;
		}

		//Devuelve el tipo de fichero AllFiles si no se encuentra correspondencia
		public static TipoFichero tipofichero___aerotri(TipoFiltro filtro, byte aerotri_code, TipoFichero nofound_retcode){
			TipoFichero[] filtroftypes=Filtros[filtro];
			foreach(TipoFichero t in filtroftypes){
				FileTypeInfo finfo;
				if(FileTypeInfos.TryGetValue(t,out finfo) && finfo.aerotri==aerotri_code) return t;
			}
			return nofound_retcode;
		}

		internal static string buildfiltro_base(TipoFichero[] filtroftypes, Dictionary<TipoFichero,NombrecortoyLargo> nombres){
			string s="";
			foreach(TipoFichero t in filtroftypes){
				string[] extensiones=FileTypeInfos[t].extensiones;

				s+=nombres[t].largo+" (*"+extensiones[0];
				for(int j=1;j<extensiones.Length;j++) s+=", *"+extensiones[j];
				s+=")|*"+extensiones[0];
				for(int j=1;j<extensiones.Length;j++) s+=";*"+extensiones[j];
				s+="|";
			}
			s+=nombres[AllFiles].largo+" (*.*)|*.*";
			return s;
		}

		public static string buildfiltro(TipoFiltro tipo, Idioma idioma){
			if(idioma>=Idioma.NIDIOMAS) idioma=Idioma.Esp;
			return buildfiltro_base(Filtros[tipo],Nombres[(byte)idioma]);
		}

		//Devuelve "" si no hay problema con la extensión, y en caso contrario el mensaje que se debe de mostrar,
		//con las opciones Continuar/Cancelar
		public static string CheckExtension(TipoFiltro tipofiltro, TipoFichero tipof, string ext, Idioma idioma){
			foreach(string s in FileTypeInfos[tipof].extensiones){
				if(ext==s) return "";
			}
			foreach(TipoFichero t in Filtros[tipofiltro]){
				if(t==tipof) continue;
				foreach(string s in FileTypeInfos[t].extensiones){
					if(ext==s){
						string ss="Ha pedido un tipo de fichero transformado "+Nombres[(byte)idioma][tipof].corto
									+" pero la extensión indicada ("+ext+") corresponde a un formato "+Nombres[(byte)idioma][t].corto
									+". ¿Es correcto y desea continuar con la transformación?";
						return ss;
					}
				}
			}
			return "";
		}
	} //Class FileTypes
}
