/******---------        INTEGER TYPE DEFINITIONS        ---------******/

#include "ATuint.h" //The three most exposed types

/* Exact-size types, as far as the compiler supports the sizes (e.g., char8_t could be 16 bits if such is
the minimum addressable storage). Both for memory saving and for compatibility between caller
and callee in their function arguments (or objects accessed therefrom). In case of 8-bit types, umint
is intended for the former use versus uint8m for the latter, but the dintinction between the two
is not clrear cut.

*  char8_t: Type por holding characters, either single-byte ones or utf-8 encoded. Must be unsigned.
It is defined in ATsystem.h.

*  bool8: A boolean type stored in one char (as opposed to bint below).

*  umint: Minimum size integer. Sizeof it equals 1 by definition (so it can be set with memset)
The C standard imposes at least 256 for its maximum value.

*  uintNm/sintNm: (u)int_leastN_t types. I assume the existence of 32-bit width integers and this header
does not provide definitions for mandatori-ly wider integers.

*  uint: The intended standard integer type for code using this headers, in particular for the ATcrt
library. These must be 32-bit width integers. (u)int32_t are not used because 2's complement is not
mandatory and padding bits are not forbidden, but the types must have a 32-bit width nonetheless.
*/
typedef unsigned char bool8;
typedef unsigned char umint;
#define uint8m uint_least8_t
#define sint8m int_least8_t
#define uint16m uint_least16_t
#define sint16m int_least16_t
//typedef uint_least32_t uint; //defined int ATuint.h
typedef int_least32_t ssint;

/*Names for integer types that are logically bool, int8 or int16 but the size of which we don't care
about (see note). In previous versions these used to be defines for the fast types, e. g. #define u8int uint_fast8_t.
However, even on the same architecture different compilers may choose different types for these,
so in order to provide compatibility between already compiled binaries and header files, to be
included by some code being compiled possibly by a different compiler, these all must have a known
size, namely, uint.
If you want the fastest possible for an object not being exposed to the outside, just use the standard (u)int_fastN_t
*/
#ifndef NO_ATCRT_FAST
	typedef uint u8int;
	typedef ssint s8int;
	typedef uint u16int;
	typedef ssint s16int;
#endif

/*The type packedbits is used for arrays of bitfields packed in an unsigned integer. This used to be
unsigned int, for this should be the type best handled by the compiler according to the C specifications.
But this caused the need of handling an extra case in many macros or functions, e.g. zeroset_<type>,
So having it like uint is much worth the hypothetical performance loss.
    If the number of bits of a packedbits is N, PACKEDBITS_SZ1 is one less than the greatest power
of 2 which is <=N, and PACKEDBITST_LOGSZ the exponent of that power of two.*/
#ifndef PACKEDBITS_SZ1
	typedef uint packedbits;
	#define PACKEDBITS_SZ1 31
	#define IPACKEDBITS_LOGSZ 5
#endif

#ifndef ATCRT_TYPES_ONLY

/*This is the type used for countdown values, as in for(cint i=n;i--;){...}. It is unsigned int
so that it can be tested against -1 without the need to cast the latter. But then testing it against
EOUIA (see below) is wrong. A user knowing about this macro will likely also know that
the use of the macros fallvar and fallthru from ATmacros.h paired with an if(FALLED(i)) is preferred
to an explicit use of cint i; for(i=n; i--; ...){ ... }, while a test against -1 is a typical programmer error.
For this reason unsigned int is preferred in place of uint. */
#define cint unsigned int


/******---------        OTHER DEFINITIONS        ---------******/

#define CHAR_BYTE (CHAR_BIT/8)

/*The macro INT_UINT means that a signed type and the corresponding unsigned type have the same
width and that if a negative value is assigned to an unsigned variable and then that variable is cast to
the corresponding signed type, the result is the same as if the original value had been assigned to a
variable of the signed type. Thus, after uint n=-2; the expression (ssint)n evaluates to -2 (this excludes
the behaviour of issuing an overflow error of some kind)
    TWO_COMP implies that, in addition, two's complement representation is used for signed types.
Note that using two's complement alone does not mean that TWO_COMP can be defined.
    These definitions enable some optimizations. The extent to which these optimizations are substantial
depends upon the use the programmer makes of this feature. They will likely not be very important in
modern compilers. At any rate, if the system_<COMPILER> file does not define these macros even if
they could be, any code using this header should compile equally right.
    The unrelated macro NO_PADDING_BITS may also be defined. This header only uses it for the
definitons of isneg and ispos. The library ATcrt does not make any use of it.

    Define NON_STD_INTEGERS in case any of the three definitions do not apply, and define the ones
that do apply as well as SIZEOF_UINT. The latter is only used once in the library ATcrt, at type_conv.h.
*/
#ifndef  NON_STD_INTEGERS
	#define INT_UINT
	#define TWO_COMP
	#define NO_PADDING_BITS
	#define SIZEOF_UINT (32/CHAR_BIT)
#endif
#if defined(TWO_COMP) && !defined(INT_UINT)
#error TWO_COMP is defined but INT_UINT not. This cannot be, TWO_COMP implies INT_UINT
#endif

/*End Of Unsigned Integer Array.
While this is not needed for setting an unsigned value to -1, it is needed for checking it against
this value, for the -1 could be promoted and just writing x==-1 would fail. Of course, the value
may be used for purposes other than signalling the end of an array of uint values. A very
common use is to mark the absence of data, or empty value, hence the synonim AT_NODATA.
    Values for integers actually or logically shorter than a uint are also defined. */
#define EOUIA ((uint)-1)
#define AT_NODATA ((uint)-1)
#ifndef Я
#define Я EOUIA //Shortcut
#define Я32 Я
#define Я16 0xFFff
#define Я8 0xFF
#endif

/*The difference of two pointers will likely be assigned to a uint, as well as the value of functions
or operations returning a size_t or ptrdiff_t, and therefore require a cast to (uint) in order to avoid
a warning. In order to distinguish these casts from other casts to uint, the name pdif is defined as
synonim to uint to be used in those cases, e.g. k=(pdif)(ptr-base); usizeof and uoffsetof are for the
same reason defined. These macros are useful even for functions taking a size_t, like malloc, to avoid
multiplying size_t-size integers; e.g., malloc(n*usizeof(float)).
*/
#define pdif uint
#define usizeof (pdif)sizeof
#define uoffsetof (pdif)offsetof
/*sizeof in uints. Used for using memcpy_uint and memeset_uint with structures*/
#ifndef uintsizeof
	#define uintsizeof(Type) (usizeof(Type)/usizeof(uint))
#endif
#ifndef uintoffsetof
	#define uintoffsetof(Type,field) (uoffsetof(Type,field)/usizeof(uint))
#endif
#ifndef ulen_array
	#define ulen_array(x) (usizeof(x)/usizeof(*(x)))
#endif

#endif
